<?php
/**
 * HUBTEL AUTO-DEBIT / RECURRING INVOICE (REAL ENDPOINTS)
 *
 * Flow:
 *  1) Create Invoice  -> returns recurringInvoiceId, requestId, otpPrefix
 *  2) Customer receives OTP -> you collect OTP from customer
 *  3) Verify Invoice  -> send otpPrefix-OTP + recurringInvoiceId + requestId
 *
 * Endpoints:
 *  Create: POST https://rip.hubtel.com/api/proxy/{HubtelAccountNumber}/create-invoice
 *  Verify: POST https://rip.hubtel.com/api/proxy/Verify-Invoice
 *  Cancel: POST https://rip.hubtel.com/api/proxy/{HubtelAccountNumber}/cancel-invoice/{recurringInvoiceId}
 */

// --------------------
// 1) SET YOUR CREDENTIALS
// --------------------
$HUBTEL_ACCOUNT_NUMBER = "YOUR_HUBTEL_ACCOUNT_NUMBER"; // {HubtelAccountNumber}
$API_ID  = "YOUR_API_ID";   // username (Hubtel API ID)
$API_KEY = "YOUR_API_KEY";  // password (Hubtel API KEY)

// NOTE: Hubtel requires PUBLIC IP WHITELISTING for Recurring Invoice endpoints.

// --------------------
// 2) ENDPOINTS
// --------------------
$CREATE_ENDPOINT = "https://rip.hubtel.com/api/proxy/{$HUBTEL_ACCOUNT_NUMBER}/create-invoice";
$VERIFY_ENDPOINT = "https://rip.hubtel.com/api/proxy/Verify-Invoice";
$CANCEL_ENDPOINT = "https://rip.hubtel.com/api/proxy/{$HUBTEL_ACCOUNT_NUMBER}/cancel-invoice"; // + /{recurringInvoiceId}

// --------------------
// 3) HELPERS
// --------------------
function hubtelAuthHeader(string $apiId, string $apiKey): string {
    return "Authorization: Basic " . base64_encode($apiId . ":" . $apiKey);
}

function postJson(string $url, array $payload, array $headers = []): array {
    $ch = curl_init($url);
    $json = json_encode($payload, JSON_UNESCAPED_SLASHES);

    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => array_merge([
            "Content-Type: application/json",
            "Accept: application/json",
            "Content-Length: " . strlen($json),
        ], $headers),
        CURLOPT_POSTFIELDS     => $json,
        CURLOPT_TIMEOUT        => 60,
    ]);

    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    if ($resp === false) {
        $err = curl_error($ch);
        curl_close($ch);
        throw new Exception("cURL error: " . $err);
    }
    curl_close($ch);

    $decoded = json_decode($resp, true);
    return [
        "httpCode" => $code,
        "raw"      => $resp,
        "json"     => is_array($decoded) ? $decoded : null,
    ];
}

/**
 * Channels (Auto-debit recurring):
 *  MTN     => mtn_gh_rec
 *  Telecel => vodafone_gh_rec
 */
function recurringChannel(string $telco): string {
    $telco = strtoupper(trim($telco));
    if ($telco === "MTN") return "mtn_gh_rec";
    if ($telco === "TELECEL") return "vodafone_gh_rec";
    throw new Exception("Unsupported telco for recurring: {$telco}");
}

// --------------------
// 4) CREATE RECURRING INVOICE (AUTO-DEBIT SETUP)
// --------------------
function createRecurringInvoice(
    string $createEndpoint,
    string $apiId,
    string $apiKey,
    string $telco,                 // "MTN" or "Telecel"
    string $customerMobileNumber,  // international: 23324xxxxxxx
    float  $recurringAmount,       // amount for each cycle
    float  $initialAmount,         // first payment (can be same as recurring)
    string $paymentInterval,       // DAILY|WEEKLY|MONTHLY|QUARTERLY
    string $startTime,             // HH:MM e.g. "10:00"
    string $callbackUrl,           // your webhook
    string $description,
    ?string $invoiceEndDate = null,// "MM/DD/YYYY HH:MM:SS" optional
    ?string $customerName  = null
): array {

    $orderDate = date("m/d/Y H:i:s"); // required format in doc

    $payload = [
        "orderDate"             => $orderDate,
        "startTime"             => $startTime,
        "paymentInterval"       => $paymentInterval,
        "customerMobileNumber"  => $customerMobileNumber,
        "paymentOption"         => "MobileMoney",
        "channel"               => recurringChannel($telco),
        "recurringAmount"       => $recurringAmount,
        "totalAmount"           => $recurringAmount,   // doc says currently same as recurringAmount
        "initialAmount"         => $initialAmount,
        "currency"              => "GHS",
        "callbackUrl"           => $callbackUrl,
        "description"           => $description,
    ];

    if (!empty($invoiceEndDate)) $payload["invoiceEndDate"] = $invoiceEndDate;
    if (!empty($customerName))   $payload["customerName"]   = $customerName;

    $headers = [ hubtelAuthHeader($apiId, $apiKey) ];
    return postJson($createEndpoint, $payload, $headers);
}

// --------------------
// 5) VERIFY RECURRING INVOICE (OTP VERIFICATION)
// --------------------
function verifyRecurringInvoice(
    string $verifyEndpoint,
    string $apiId,
    string $apiKey,
    string $recurringInvoiceId,
    string $requestId,
    string $otpPrefix,     // from create response
    string $otp4Digits     // customer OTP digits
): array {

    $payload = [
        "recurringInvoiceId" => $recurringInvoiceId,
        "requestId"          => $requestId,
        "otpCode"            => $otpPrefix . "-" . $otp4Digits, // e.g. OQDM-9514
    ];

    $headers = [ hubtelAuthHeader($apiId, $apiKey) ];
    return postJson($verifyEndpoint, $payload, $headers);
}

// --------------------
// 6) EXAMPLE USAGE
// --------------------

// STEP A: Create
try {
    $createRes = createRecurringInvoice(
        $CREATE_ENDPOINT,
        $API_ID,
        $API_KEY,
        "MTN",                 // or "Telecel"
        "233244000000",
        10.00,                 // recurringAmount
        10.00,                 // initialAmount
        "MONTHLY",             // paymentInterval
        "10:00",               // startTime
        "https://yourdomain.com/hubtel/recurring-callback.php",
        "Policy premium auto-debit",
        null,                  // invoiceEndDate optional e.g. "12/31/2026 23:59:00"
        "Martin Danso"
    );

    // Print create response
    header("Content-Type: application/json");
    echo json_encode([
        "step" => "CREATE",
        "result" => $createRes
    ], JSON_PRETTY_PRINT);

    /**
     * After CREATE, you should get (in json):
     *  data.recurringInvoiceId
     *  data.requestId
     *  data.otpPrefix
     *
     * Then collect customer OTP and call verifyRecurringInvoice().
     */

} catch (Exception $e) {
    http_response_code(500);
    header("Content-Type: application/json");
    echo json_encode(["error" => true, "message" => $e->getMessage()], JSON_PRETTY_PRINT);
}
