<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

include("database.php"); // must create $con (mysqli)
date_default_timezone_set("Africa/Accra");

$path = $_SERVER['REQUEST_URI'] ?? 'check_age_range.php';

/* =========================
   LOG FUNCTION
   - userindex can be any identifier you want.
   - For this script we log userindex as the requester (if provided), else "UNKNOWN".
========================= */
function log_activity($con, $userindex, $caption, $path, $data_inf) {
    if (!isset($con) || !$con) return;

    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        $u = (string)$userindex;
        $c = (string)$caption;
        $p = (string)$path;
        $d = (string)$data_inf;

        mysqli_stmt_bind_param($stmt, "ssss", $u, $c, $p, $d);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

// ---------- METHOD CHECK ----------
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    log_activity($con, "UNKNOWN", "INVALID REQUEST METHOD", $path, json_encode([
        "method" => $_SERVER["REQUEST_METHOD"] ?? ""
    ]));
    echo json_encode(["ok" => false]);
    exit;
}

if (!isset($con) || !$con) {
    // Cannot log into DB if DB is down
    echo json_encode(["ok" => false]);
    exit;
}

// ---- Read JSON or POST ----
$raw  = file_get_contents("php://input");
$data = json_decode($raw, true);
$isJson = is_array($data);

$life_type     = trim($isJson ? ($data["life_type"] ?? "") : ($_POST["life_type"] ?? ""));
$date_of_birth = trim($isJson ? ($data["date_of_birth"] ?? "") : ($_POST["date_of_birth"] ?? ""));

// Optional requester identifier to log (if you pass it). Example: mobile_money_number
$requester = trim($isJson ? ($data["userindex"] ?? ($data["mobile_money_number"] ?? "")) : ($_POST["userindex"] ?? ($_POST["mobile_money_number"] ?? "")));
if ($requester === "") $requester = "UNKNOWN";

// Log request received
log_activity($con, $requester, "REQUEST RECEIVED", $path, json_encode([
    "life_type" => $life_type,
    "date_of_birth" => $date_of_birth
]));

// ---- Basic validation ----
if ($life_type === "" || $date_of_birth === "") {
    log_activity($con, $requester, "VALIDATION FAILED (missing inputs)", $path, json_encode([
        "life_type" => $life_type,
        "date_of_birth" => $date_of_birth
    ]));
    echo json_encode(["ok" => false]);
    exit;
}

// ---- Validate DOB ----
$dob = DateTime::createFromFormat("Y-m-d", $date_of_birth);
if (!$dob) {
    log_activity($con, $requester, "VALIDATION FAILED (invalid dob format)", $path, json_encode([
        "date_of_birth" => $date_of_birth,
        "expected_format" => "YYYY-MM-DD"
    ]));
    echo json_encode(["ok" => false]);
    exit;
}

$today = new DateTime();
if ($dob > $today) {
    log_activity($con, $requester, "VALIDATION FAILED (dob in future)", $path, json_encode([
        "date_of_birth" => $date_of_birth
    ]));
    echo json_encode(["ok" => false]);
    exit;
}

// ---- Calculate age ----
$age = (int)$today->diff($dob)->y;

log_activity($con, $requester, "AGE CALCULATED", $path, json_encode([
    "age" => $age,
    "life_type" => $life_type
]));

// ---- Check age against rules ----
$sql = "
    SELECT 1
    FROM waiting_period_rules
    WHERE life_type = ?
      AND ? BETWEEN min_age AND max_age
    LIMIT 1
";

$stmt = mysqli_prepare($con, $sql);
if (!$stmt) {
    log_activity($con, $requester, "DB PREPARE FAILED", $path, mysqli_error($con));
    echo json_encode(["ok" => false]);
    exit;
}

mysqli_stmt_bind_param($stmt, "si", $life_type, $age);

if (!mysqli_stmt_execute($stmt)) {
    log_activity($con, $requester, "DB EXECUTE FAILED", $path, mysqli_stmt_error($stmt));
    mysqli_stmt_close($stmt);
    echo json_encode(["ok" => false]);
    exit;
}

$res = mysqli_stmt_get_result($stmt);
mysqli_stmt_close($stmt);

// ---- Final response ----
if ($res && mysqli_num_rows($res) > 0) {
    log_activity($con, $requester, "AGE CHECK PASSED", $path, json_encode([
        "life_type" => $life_type,
        "age" => $age,
        "result" => "ok"
    ]));
    echo json_encode(["ok" => true]);
} else {
    log_activity($con, $requester, "AGE CHECK FAILED", $path, json_encode([
        "life_type" => $life_type,
        "age" => $age,
        "result" => "failed"
    ]));
    echo json_encode(["ok" => false]);
}

?>