<?php
// mandate_create.php
// Safe api_num increment (transaction + row lock)
// NO forcing of amountToDebit
// ✅ Adds logs into users_logs when running mandate creation

/**
 * Log into users_logs table (safe: does not break main flow if logging fails)
 */
function add_user_log(
    mysqli $con,
    int $userindex,
    string $caption,
    string $path,
    string $data_inf
): void {
    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    $stmt = $con->prepare($sql);
    if (!$stmt) return;

    $stmt->bind_param("isss", $userindex, $caption, $path, $data_inf);
    $stmt->execute();
    $stmt->close();
}

function create_mandate(
    mysqli $con,
    int $userindex,                 // ✅ added (for users_logs)
    string $momo_network,
    string $phoneNumber,
    string $temp_policy_num,
    string $frequencyType,
    string $amountToDebit = '',
    string $path = 'mandate_create.php' // ✅ optional path for logging
): array {

    // Start log (inputs)
    add_user_log(
        $con,
        $userindex,
        "Mandate Creation - START",
        $path,
        json_encode([
            "momo_network" => $momo_network,
            "phoneNumber" => $phoneNumber,
            "temp_policy_num" => $temp_policy_num,
            "frequencyType" => $frequencyType,
            "amountToDebit" => $amountToDebit
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    );

    // -----------------------------------
    // 1) SAFELY GET + INCREMENT api_num
    // -----------------------------------
    $api_num  = null;
    $api_numn = null;

    try {
        $con->begin_transaction();

        $stmt = $con->prepare(
            "SELECT api_num FROM generate_idn 
             WHERE generateindex='1' 
             LIMIT 1 FOR UPDATE"
        );
        if (!$stmt || !$stmt->execute()) {
            throw new Exception("Failed to read api_num");
        }

        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();

        if (!$row || !isset($row['api_num'])) {
            throw new Exception("api_num not found");
        }

        $api_num  = (int)$row['api_num'];
        $api_numn = $api_num + 1;

        $stmt2 = $con->prepare(
            "UPDATE generate_idn SET api_num=? WHERE generateindex='1'"
        );
        if (!$stmt2) {
            throw new Exception("Prepare update failed");
        }
        $stmt2->bind_param("i", $api_numn);
        if (!$stmt2->execute()) {
            throw new Exception("Update api_num failed");
        }
        $stmt2->close();

        $con->commit();

        add_user_log(
            $con,
            $userindex,
            "Mandate Creation - api_num increment OK",
            $path,
            json_encode(["api_num_used" => $api_num, "api_num_next" => $api_numn], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

    } catch (Exception $e) {
        $con->rollback();

        add_user_log(
            $con,
            $userindex,
            "Mandate Creation - api_num increment FAILED",
            $path,
            json_encode(["error" => $e->getMessage()], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

        return [
            "ok" => false,
            "error" => "api_num increment failed: " . $e->getMessage()
        ];
    }

    // -----------------------------------
    // 2) Dates
    // -----------------------------------
    $insured_date_ans  = date('Y-m-d');
    $maturity_date_ans = date('Y-m-d', strtotime($insured_date_ans . ' +3 years'));

    // -----------------------------------
    // 3) Network mapping (NO forcing)
    // -----------------------------------
    if ($momo_network === 'MTN') {
        $network  = 'MTN';
        $debitDay = '30';
    } elseif ($momo_network === 'Telecel') {
        $network  = 'VODAFONE';
        $debitDay = '11';
    } else {
        add_user_log(
            $con,
            $userindex,
            "Mandate Creation - FAILED (Unsupported Network)",
            $path,
            json_encode(["momo_network" => $momo_network], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

        return [
            "ok" => false,
            "error" => "Unsupported momo_network: {$momo_network}"
        ];
    }

    // -----------------------------------
    // 4) Build request payload
    // -----------------------------------
    $merchantId = '1664';
    $productId  = '3';
    $apiKey     = 'c1SpFDs1ONN0SjawWMkYawwOAWAAu2Lb';

    $thirdPartyReferenceNo = $api_num . "-" . $temp_policy_num;

    $payload = [
        "merchantId"            => $merchantId,
        "productId"             => $productId,
        "clientPhone"           => $phoneNumber,
        "amountToDebit"         => (string)$amountToDebit, // AS-IS
        "network"               => $network,
        "frequencyType"         => $frequencyType,
        "frequency"             => "1",
        "startDate"             => $insured_date_ans,
        "endDate"               => $maturity_date_ans,
        "debitDay"              => $debitDay,
        "apiKey"                => $apiKey,
        "thirdPartyReferenceNo" => $thirdPartyReferenceNo
    ];

    add_user_log(
        $con,
        $userindex,
        "Mandate Creation - REQUEST",
        $path,
        json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    );

    // -----------------------------------
    // 5) cURL call
    // -----------------------------------
    $ch = curl_init("https://ddv15-do.transflowitc.com/create/mandate");
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_SSL_VERIFYPEER => 0,
        CURLOPT_CUSTOMREQUEST  => "POST",
        CURLOPT_POSTFIELDS     => json_encode($payload),
        CURLOPT_HTTPHEADER     => [
            "Content-Type: application/json",
            "Cache-Control: no-cache"
        ],
    ]);

    $response = curl_exec($ch);
    $err      = curl_error($ch);
    $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err) {
        add_user_log(
            $con,
            $userindex,
            "Mandate Creation - CURL ERROR",
            $path,
            json_encode(["http_code" => $httpCode, "curl_error" => $err], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

        return [
            "ok" => false,
            "error" => "cURL Error: {$err}",
            "http_code" => $httpCode
        ];
    }

    // Log raw response (shortened to keep DB safe)
    add_user_log(
        $con,
        $userindex,
        "Mandate Creation - RAW RESPONSE",
        $path,
        json_encode(["http_code" => $httpCode, "raw" => substr((string)$response, 0, 2000)], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    );

    $decoded = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        add_user_log(
            $con,
            $userindex,
            "Mandate Creation - FAILED (Invalid JSON)",
            $path,
            json_encode(["http_code" => $httpCode, "raw" => substr((string)$response, 0, 2000)], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

        return [
            "ok" => false,
            "error" => "Invalid JSON from mandate API",
            "http_code" => $httpCode,
            "raw_response" => substr((string)$response, 0, 1200),
            "request" => $payload
        ];
    }

    // -----------------------------------
    // 6) Success
    // -----------------------------------
    add_user_log(
        $con,
        $userindex,
        "Mandate Creation - SUCCESS",
        $path,
        json_encode(["http_code" => $httpCode, "response" => $decoded], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    );

    return [
        "ok" => true,
        "http_code" => $httpCode,
        "api_num_used" => $api_num,
        "api_num_next" => $api_numn,
        "insured_date_ans" => $insured_date_ans,
        "maturity_date_ans" => $maturity_date_ans,
        "thirdPartyReferenceNo" => $thirdPartyReferenceNo,
        "request" => $payload,
        "response" => $decoded
    ];
}