<?php
// ussd.php (FULL SCRIPT)
// ✅ Menu supports:
//    1. Register
//    2. Pay Premium
//    3. Stop Auto Deduct
//    4. Claims
//    5. Policy Status
//    6. T&C
//
// ✅ Back-to-menu key: 0
// ✅ FIXED: If gateway sends accumulated text like 2*0*3*4, script now treats it as NEW journey (3*4)
//          by slicing inputs after the LAST 0 and re-routing correctly.
//
// - Uses MSISDN (phoneNumber) as mobile_money_number
// - Premium API "no" is "" when no dependents, else "1" or "2"
// - Fixes $sumAssured variable usage
// - If any API fails with HTTP 200 but ok=false, prints FULL JSON response

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
    header("Content-Type: text/plain; charset=utf-8");
    echo "END FATAL: " . $e['message'] . " in " . $e['file'] . ":" . $e['line'];
  }
});

require_once "database.php";
require_once "helpers.php";

/**
 * ===== CONFIG: Your API endpoints =====
 */
$API_BASE   = "http://localhost/abrabo_pa_ussd/api";

// Register APIs
$SIGNUP_API      = $API_BASE . "/signup.php";
$AGE_API         = $API_BASE . "/age.php";
$PRODUCT_API     = $API_BASE . "/product.php";
$SUM_ASSURED_API = $API_BASE . "/sum_assured.php";
$PREMIUM_API     = $API_BASE . "/get_premium.php";

// ✅ Shared lookup
$LOOKUP_API      = $API_BASE . "/look_up.php";

// ✅ Pay Premium API
$PAY_PREMIUM_API = $API_BASE . "/pay_premium.php";

// ✅ Stop Auto Deduct API
$STOP_AUTODEBIT_API = $API_BASE . "/stop_autodebit.php";

// ✅ Claims Signup API
$CLAIMS_SIGNUP_API = $API_BASE . "/claims_signup.php";

// ✅ Policy Status API (GET ?policy_number=...)
$POLICY_STATUS_API = $API_BASE . "/policy_status.php";

// ✅ Terms & Conditions API (POST { mobile_no })
$TERMS_API = $API_BASE . "/terms_condition.php";

/**
 * Gateway params
 */
$sessionId   = ussd_clean($_POST["sessionId"] ?? $_REQUEST["sessionId"] ?? "");
$phoneNumber = ussd_clean($_POST["phoneNumber"] ?? $_REQUEST["phoneNumber"] ?? "");
$text        = ussd_clean($_POST["text"] ?? $_REQUEST["text"] ?? "");
$serviceCode = ussd_clean($_POST["serviceCode"] ?? $_REQUEST["serviceCode"] ?? "*741*50#");

if ($sessionId === "" || $phoneNumber === "") {
  ussd_response("END", "Invalid request: sessionId/phoneNumber missing.");
}

$inputs = ussd_parse_text($text);

/**
 * Helper: reload draft anytime you need current state
 */
function reload_draft($con, $sessionId) {
  return db_fetch_one($con, "SELECT * FROM ussd_reg_drafts WHERE session_id=? LIMIT 1", "s", [$sessionId]) ?? [];
}

/**
 * ✅ HOME MENU (single source of truth)
 */
function home_menu_text() {
  return
    "Welcome to Impact Life\n" .
    "======================\n\n" .
    "1. Register\n" .
    "2. Pay Premium\n" .
    "3. Stop Auto Deduct\n" .
    "4. Claims\n" .
    "5. Policy Status\n" .
    "6. T&C";
}

function reset_state_only($con, $sessionId) {
  draft_update($con, $sessionId, [
    "step" => "START",
    "flow" => "",
    "pay_policies_json" => null,
    "pay_policy_number" => null,
    "total_premium" => null,
    "momo_network" => null,
    "stop_policies_json" => null,
    "stop_policy_number" => null,
    "claim_policies_json" => null,
    "claim_policy_number" => null,
    "claim_full_name_deceased" => null,
    "claim_type" => null,
    "policy_policies_json" => null,
    "policy_policy_number" => null
  ]);
}

function go_home($con, $sessionId) {
  reset_state_only($con, $sessionId);
  ussd_response("CON", home_menu_text());
}

function is_back_choice($val) {
  $v = trim((string)$val);
  return ($v === "0");
}

/**
 * Helper: show products ONCE (sets step SELECT_PRODUCT always)
 */
function show_products($con, $sessionId, $PRODUCT_API) {
  $prodResp = get_json($PRODUCT_API);
  if (!($prodResp["ok"] ?? false)) ussd_response("END", "Unable to load products.");

  $products = $prodResp["products"] ?? [];
  if (count($products) === 0) ussd_response("END", "No products available.");

  $menu = "Select Product:\n";
  foreach ($products as $i => $p) {
    $n = $i + 1;
    $menu .= "{$n}. {$p["product_name"]}\n";
  }

  draft_update($con, $sessionId, ["step" => "SELECT_PRODUCT", "flow" => "REGISTER"]);
  ussd_response("CON", trim($menu));
}

/**
 * Ensure session exists
 */
db_exec(
  $con,
  "INSERT IGNORE INTO ussd_sessions (session_id, msisdn, service_code, last_text) VALUES (?, ?, ?, ?)",
  "ssss",
  [$sessionId, $phoneNumber, $serviceCode, $text]
);

db_exec(
  $con,
  "UPDATE ussd_sessions SET last_text=? WHERE session_id=?",
  "ss",
  [$text, $sessionId]
);

/**
 * Ensure draft exists
 */
db_exec(
  $con,
  "INSERT IGNORE INTO ussd_reg_drafts (session_id, msisdn) VALUES (?, ?)",
  "ss",
  [$sessionId, $phoneNumber]
);

$draft = reload_draft($con, $sessionId);
$step  = $draft["step"] ?? "START";
$flow  = $draft["flow"] ?? "";

/**
 * HOME MENU
 */
if (count($inputs) === 0) {
  draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
  ussd_response("CON", home_menu_text());
}

$menuChoice = $inputs[0] ?? "";
$lastInput  = $inputs[count($inputs) - 1] ?? "";

/**
 * ✅ FIXED Back-to-menu logic:
 * If "0" appears anywhere (e.g. 2*0*3*4), treat everything after LAST 0 as a NEW journey.
 */
$lastZeroPos = -1;
for ($i = count($inputs) - 1; $i >= 0; $i--) {
  if (is_back_choice($inputs[$i])) { $lastZeroPos = $i; break; }
}

if ($lastZeroPos !== -1) {
  // Always reset stored state when 0 occurs
  reset_state_only($con, $sessionId);

  // If 0 is the last input => show home menu now
  if ($lastZeroPos === count($inputs) - 1) {
    go_home($con, $sessionId);
  }

  // If user continued typing after 0 => treat remainder as new path
  $inputs = array_values(array_slice($inputs, $lastZeroPos + 1));

  if (count($inputs) === 0) {
    go_home($con, $sessionId);
  }

  // Recompute routing vars
  $menuChoice = $inputs[0] ?? "";
  $lastInput  = $inputs[count($inputs) - 1] ?? "";
}

/**
 * ==========================
 * ✅ T&C FLOW (6)
 * ==========================
 */
if ($menuChoice === "6") {

  $msisdn = (string)$phoneNumber;

  $resp = post_json($TERMS_API, [
    "mobile_no" => $msisdn
  ]);

  draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);

  if (!($resp["ok"] ?? false)) {
    $dump = json_encode($resp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ussd_response("END", "Unable to send T&C.\n{$dump}");
  }

  $msg = (string)($resp["message"] ?? "Request submitted.");
  ussd_response("END", "T&C\n{$msg}\nCheck your SMS.");
}

/**
 * ==========================
 * ✅ POLICY STATUS FLOW (5) + Back to Menu
 * ==========================
 */
if ($menuChoice === "5") {

  if ($flow !== "POLICY") {
    draft_update($con, $sessionId, ["flow" => "POLICY", "step" => "POLICY_SELECT_POLICY"]);
    $draft = reload_draft($con, $sessionId);
  }

  $draft = reload_draft($con, $sessionId);
  $policies = json_decode($draft["policy_policies_json"] ?? "[]", true) ?: [];

  if (count($policies) === 0) {

    $lk = post_json($LOOKUP_API, ["mobile_money_number" => (string)$phoneNumber]);

    if (!($lk["ok"] ?? false)) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy found for this number.");
    }

    $data  = $lk["data"] ?? null;
    $items = [];

    if (is_array($data) && !isset($data[0])) {
      if (!empty($data["policy_number"])) $items[] = $data;
    }

    if (is_array($data) && isset($data[0])) {
      foreach ($data as $row) {
        if (is_array($row) && !empty($row["policy_number"])) $items[] = $row;
      }
    }

    if (count($items) === 0) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy number found.");
    }

    draft_update($con, $sessionId, [
      "policy_policies_json" => json_encode($items, JSON_UNESCAPED_UNICODE),
      "step" => "POLICY_SELECT_POLICY",
      "flow" => "POLICY"
    ]);

    $draft = reload_draft($con, $sessionId);
    $policies = json_decode($draft["policy_policies_json"] ?? "[]", true) ?: [];
  }

  if (($draft["step"] ?? "") === "POLICY_SELECT_POLICY") {

    $menu = "Policy Status\nSelect Policy Number:\n";
    foreach ($policies as $i => $row) {
      $n = $i + 1;
      $menu .= "{$n}. " . ($row["policy_number"] ?? "") . "\n";
    }
    $menu .= "0. Back to Menu";

    if (count($inputs) < 2) {
      ussd_response("CON", trim($menu));
    }

    if (is_back_choice($inputs[1] ?? "")) {
      go_home($con, $sessionId);
    }

    $choice = (int)($inputs[1] ?? 0);
    if ($choice <= 0 || $choice > count($policies)) {
      ussd_response("CON", trim($menu));
    }

    $sel = $policies[$choice - 1];
    $policyNo = (string)($sel["policy_number"] ?? "");

    if ($policyNo === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Policy number missing. Please try again.");
    }

    $url  = $POLICY_STATUS_API . "?" . http_build_query(["policy_number" => $policyNo]);
    $resp = get_json($url);

    draft_update($con, $sessionId, [
      "step" => "START",
      "flow" => "",
      "policy_policies_json" => null,
      "policy_policy_number" => null
    ]);

    if (!($resp["ok"] ?? false)) {
      $dump = json_encode($resp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
      ussd_response("END", "Unable to load policy status.\n{$dump}");
    }

    $pno    = (string)($resp["policy_number"] ?? $policyNo);
    $status = (string)($resp["status"] ?? "UNKNOWN");
    $cover  = (string)($resp["cover"] ?? "");
    $prem   = (string)($resp["premium"] ?? "");
    $paid   = (string)($resp["paid"] ?? "");

    $msg = "Policy: {$pno}\nStatus: {$status}";
    if ($cover !== "") $msg .= "\nCover: GHS {$cover}";
    if ($prem !== "")  $msg .= "\nPremium: {$prem}";
    if ($paid !== "")  $msg .= "\nPaid: {$paid}";

    ussd_response("END", $msg);
  }

  ussd_response("END", "Session error. Please try again.");
}

/**
 * ==========================
 * ✅ CLAIMS FLOW (4) + Back to Menu
 * ==========================
 */
if ($menuChoice === "4") {

  if ($flow !== "CLAIM") {
    draft_update($con, $sessionId, ["flow" => "CLAIM", "step" => "CLAIM_SELECT_POLICY"]);
    $draft = reload_draft($con, $sessionId);
  }

  $draft = reload_draft($con, $sessionId);
  $policies = json_decode($draft["claim_policies_json"] ?? "[]", true) ?: [];

  if (count($policies) === 0) {
    $lk = post_json($LOOKUP_API, ["mobile_money_number" => (string)$phoneNumber]);

    if (!($lk["ok"] ?? false)) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy found for this number.");
    }

    $data  = $lk["data"] ?? null;
    $items = [];

    if (is_array($data) && !isset($data[0])) {
      if (!empty($data["policy_number"])) $items[] = $data;
    }
    if (is_array($data) && isset($data[0])) {
      foreach ($data as $row) {
        if (is_array($row) && !empty($row["policy_number"])) $items[] = $row;
      }
    }

    if (count($items) === 0) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy number found.");
    }

    draft_update($con, $sessionId, [
      "claim_policies_json" => json_encode($items, JSON_UNESCAPED_UNICODE),
      "step" => "CLAIM_SELECT_POLICY",
      "flow" => "CLAIM"
    ]);

    $draft = reload_draft($con, $sessionId);
    $policies = json_decode($draft["claim_policies_json"] ?? "[]", true) ?: [];
  }

  if (($draft["step"] ?? "") === "CLAIM_SELECT_POLICY") {

    $menu = "Claims\nSelect Policy Number:\n";
    foreach ($policies as $i => $row) {
      $n = $i + 1;
      $menu .= "{$n}. " . ($row["policy_number"] ?? "") . "\n";
    }
    $menu .= "0. Back to Menu";

    if (count($inputs) < 2) {
      ussd_response("CON", trim($menu));
    }

    if (is_back_choice($inputs[1] ?? "")) {
      go_home($con, $sessionId);
    }

    $choice = (int)($inputs[1] ?? 0);
    if ($choice <= 0 || $choice > count($policies)) {
      ussd_response("CON", trim($menu));
    }

    $sel = $policies[$choice - 1];
    $policyNo = (string)($sel["policy_number"] ?? "");

    if ($policyNo === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Policy number missing. Please try again.");
    }

    draft_update($con, $sessionId, [
      "claim_policy_number" => $policyNo,
      "step" => "CLAIM_ASK_DECEASED_NAME",
      "flow" => "CLAIM"
    ]);

    ussd_response("CON", "Claims\nEnter Full Name of Deceased:\n0. Back to Menu");
  }

  if (($draft["step"] ?? "") === "CLAIM_ASK_DECEASED_NAME") {

    if (count($inputs) < 3) {
      ussd_response("CON", "Claims\nEnter Full Name of Deceased:\n0. Back to Menu");
    }

    if (is_back_choice($inputs[2] ?? "")) {
      go_home($con, $sessionId);
    }

    $name = trim((string)($inputs[2] ?? ""));
    if ($name === "") {
      ussd_response("CON", "Claims\nEnter Full Name of Deceased:\n0. Back to Menu");
    }

    draft_update($con, $sessionId, [
      "claim_full_name_deceased" => $name,
      "step" => "CLAIM_ASK_TYPE",
      "flow" => "CLAIM"
    ]);

    ussd_response(
      "CON",
      "Select Claim Type:\n" .
        "1. Death (Main)\n" .
        "2. Death (Parent/Parent-in-law)\n" .
        "3. Accidental\n" .
        "4. Critical Illness\n" .
        "0. Back to Menu"
    );
  }

  if (($draft["step"] ?? "") === "CLAIM_ASK_TYPE") {

    if (count($inputs) < 4) {
      ussd_response(
        "CON",
        "Select Claim Type:\n" .
          "1. Death (Main)\n" .
          "2. Death (Parent/Parent-in-law)\n" .
          "3. Accidental\n" .
          "4. Critical Illness\n" .
          "0. Back to Menu"
      );
    }

    if (is_back_choice($inputs[3] ?? "")) {
      go_home($con, $sessionId);
    }

    $t = (string)($inputs[3] ?? "");

    $typeMap = [
      "1" => "DEATH",
      "2" => "DEATH_PARENT",
      "3" => "ACCIDENTAL",
      "4" => "CRITICAL_ILLNESS"
    ];

    if (!isset($typeMap[$t])) {
      ussd_response(
        "CON",
        "Invalid selection.\nSelect Claim Type:\n" .
          "1. Death (Main)\n" .
          "2. Death (Parent/Parent-in-law)\n" .
          "3. Accidental\n" .
          "4. Critical Illness\n" .
          "0. Back to Menu"
      );
    }

    draft_update($con, $sessionId, [
      "claim_type" => $typeMap[$t],
      "step" => "CLAIM_ASK_DATE",
      "flow" => "CLAIM"
    ]);

    ussd_response("CON", "Enter Date of Incident (YYYY-MM-DD):\n0. Back to Menu");
  }

  if (($draft["step"] ?? "") === "CLAIM_ASK_DATE") {

    if (count($inputs) < 5) {
      ussd_response("CON", "Enter Date of Incident (YYYY-MM-DD):\n0. Back to Menu");
    }

    if (is_back_choice($inputs[4] ?? "")) {
      go_home($con, $sessionId);
    }

    $date = trim((string)($inputs[4] ?? ""));
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
      ussd_response("CON", "Invalid format. Enter Date of Incident as YYYY-MM-DD:\n0. Back to Menu");
    }

    $draft = reload_draft($con, $sessionId);

    $policyNo   = (string)($draft["claim_policy_number"] ?? "");
    $deceased   = (string)($draft["claim_full_name_deceased"] ?? "");
    $claimType  = (string)($draft["claim_type"] ?? "");

    if ($policyNo === "" || $deceased === "" || $claimType === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Missing claim details. Please try again.");
    }

    $resp = post_json($CLAIMS_SIGNUP_API, [
      "mobile_money_number" => (string)$phoneNumber,
      "policy_number"       => $policyNo,
      "full_name_deceased"  => $deceased,
      "claim_type"          => $claimType,
      "date_of_incident"    => $date
    ]);

    draft_update($con, $sessionId, [
      "step" => "START",
      "flow" => "",
      "claim_policies_json" => null,
      "claim_policy_number" => null,
      "claim_full_name_deceased" => null,
      "claim_type" => null
    ]);

    if (!($resp["ok"] ?? false)) {
      $dump = json_encode($resp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
      ussd_response("END", "Claim signup failed.\n{$dump}");
    }

    $status  = (string)($resp["status"] ?? "OK");
    $claimId = (string)($resp["claim_id"] ?? "");
    $claimNo = (string)($resp["claim_number"] ?? "");

    $msg = "Claim submitted.\nStatus: {$status}";
    if ($claimNo !== "") $msg .= "\nClaim No: {$claimNo}";
    if ($claimId !== "") $msg .= "\nClaim ID: {$claimId}";

    ussd_response("END", $msg);
  }

  ussd_response("END", "Session error. Please try again.");
}

/**
 * ==========================
 * ✅ STOP AUTO DEDUCT FLOW (3) + Back to Menu
 * ==========================
 */
if ($menuChoice === "3") {

  if ($flow !== "STOP") {
    draft_update($con, $sessionId, ["flow" => "STOP", "step" => "STOP_SELECT_POLICY"]);
    $draft = reload_draft($con, $sessionId);
  }

  $draft = reload_draft($con, $sessionId);
  $policies = json_decode($draft["stop_policies_json"] ?? "[]", true) ?: [];

  if (count($policies) === 0) {
    $lk = post_json($LOOKUP_API, ["mobile_money_number" => (string)$phoneNumber]);

    if (!($lk["ok"] ?? false)) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy found for this number.");
    }

    $data  = $lk["data"] ?? null;
    $items = [];

    if (is_array($data) && !isset($data[0])) {
      if (!empty($data["policy_number"])) $items[] = $data;
    }
    if (is_array($data) && isset($data[0])) {
      foreach ($data as $row) {
        if (is_array($row) && !empty($row["policy_number"])) $items[] = $row;
      }
    }

    if (count($items) === 0) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy number found.");
    }

    draft_update($con, $sessionId, [
      "stop_policies_json" => json_encode($items, JSON_UNESCAPED_UNICODE),
      "step" => "STOP_SELECT_POLICY",
      "flow" => "STOP"
    ]);

    $draft = reload_draft($con, $sessionId);
    $policies = json_decode($draft["stop_policies_json"] ?? "[]", true) ?: [];
  }

  if (($draft["step"] ?? "") === "STOP_SELECT_POLICY") {

    $menu = "Stop Auto Deduct\nSelect Policy Number:\n";
    foreach ($policies as $i => $row) {
      $n = $i + 1;
      $menu .= "{$n}. " . ($row["policy_number"] ?? "") . "\n";
    }
    $menu .= "0. Back to Menu";

    if (count($inputs) < 2) {
      ussd_response("CON", trim($menu));
    }

    if (is_back_choice($inputs[1] ?? "")) {
      go_home($con, $sessionId);
    }

    $choice = (int)($inputs[1] ?? 0);
    if ($choice <= 0 || $choice > count($policies)) {
      ussd_response("CON", trim($menu));
    }

    $sel = $policies[$choice - 1];
    $policyNo = (string)($sel["policy_number"] ?? "");

    if ($policyNo === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Policy number missing. Please try again.");
    }

    draft_update($con, $sessionId, [
      "stop_policy_number" => $policyNo,
      "step" => "STOP_CONFIRM",
      "flow" => "STOP"
    ]);

    ussd_response(
      "CON",
      "Confirm Stop Auto Deduct\n" .
        "Policy: {$policyNo}\n" .
        "MSISDN: {$phoneNumber}\n\n" .
        "1. Confirm\n" .
        "2. Cancel\n" .
        "0. Back to Menu"
    );
  }

  if (($draft["step"] ?? "") === "STOP_CONFIRM") {
    $c = $lastInput;

    if ($c === "0") go_home($con, $sessionId);

    if ($c === "2") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Cancelled.");
    }

    if ($c !== "1") {
      ussd_response("CON", "1. Confirm\n2. Cancel\n0. Back to Menu");
    }

    $draft = reload_draft($con, $sessionId);
    $policyNo = (string)($draft["stop_policy_number"] ?? "");

    if ($policyNo === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Missing policy number. Please try again.");
    }

    $stopResp = post_json($STOP_AUTODEBIT_API, [
      "policy_number" => $policyNo,
      "mobil_no"      => (string)$phoneNumber
    ]);

    draft_update($con, $sessionId, [
      "step" => "START",
      "flow" => "",
      "stop_policies_json" => null,
      "stop_policy_number" => null
    ]);

    if (!($stopResp["ok"] ?? false)) {
      $dump = json_encode($stopResp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
      ussd_response("END", "Stop Auto Deduct failed.\n{$dump}");
    }

    $msg = (string)($stopResp["message"] ?? "Request submitted.");
    ussd_response("END", "Request submitted.\n{$msg}");
  }

  ussd_response("END", "Session error. Please try again.");
}

/**
 * ==========================
 * ✅ PAY PREMIUM FLOW (2) + Back to Menu
 * ==========================
 */
if ($menuChoice === "2") {

  if ($flow !== "PAY") {
    draft_update($con, $sessionId, ["flow" => "PAY", "step" => "PAY_SELECT_POLICY"]);
    $draft = reload_draft($con, $sessionId);
  }

  $draft = reload_draft($con, $sessionId);
  $policies = json_decode($draft["pay_policies_json"] ?? "[]", true) ?: [];

  if (count($policies) === 0) {
    $lk = post_json($LOOKUP_API, ["mobile_money_number" => (string)$phoneNumber]);

    if (!($lk["ok"] ?? false)) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy found for this number.");
    }

    $data  = $lk["data"] ?? null;
    $items = [];

    if (is_array($data) && !isset($data[0])) {
      if (!empty($data["policy_number"])) $items[] = $data;
    }
    if (is_array($data) && isset($data[0])) {
      foreach ($data as $row) {
        if (is_array($row) && !empty($row["policy_number"])) $items[] = $row;
      }
    }

    if (count($items) === 0) {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "No policy number found.");
    }

    draft_update($con, $sessionId, [
      "pay_policies_json" => json_encode($items, JSON_UNESCAPED_UNICODE),
      "step" => "PAY_SELECT_POLICY",
      "flow" => "PAY"
    ]);

    $draft = reload_draft($con, $sessionId);
    $policies = json_decode($draft["pay_policies_json"] ?? "[]", true) ?: [];
  }

  if (($draft["step"] ?? "") === "PAY_SELECT_POLICY") {

    $menu = "Pay Premium\nSelect Policy Number:\n";
    foreach ($policies as $i => $row) {
      $n = $i + 1;
      $menu .= "{$n}. " . ($row["policy_number"] ?? "") . "\n";
    }
    $menu .= "0. Back to Menu";

    if (count($inputs) < 2) {
      ussd_response("CON", trim($menu));
    }

    if (is_back_choice($inputs[1] ?? "")) {
      go_home($con, $sessionId);
    }

    $choice = (int)($inputs[1] ?? 0);
    if ($choice <= 0 || $choice > count($policies)) {
      ussd_response("CON", trim($menu));
    }

    $sel = $policies[$choice - 1];

    $policyNo = (string)($sel["policy_number"] ?? "");
    $premium  = (string)($sel["total_premium"] ?? "");
    $net      = (string)($sel["momo_network"] ?? "");

    if ($policyNo === "" || $premium === "" || $net === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Policy details incomplete. Contact support.");
    }

    draft_update($con, $sessionId, [
      "pay_policy_number" => $policyNo,
      "total_premium"     => $premium,
      "momo_network"      => $net,
      "step"              => "PAY_CONFIRM",
      "flow"              => "PAY"
    ]);

    ussd_response(
      "CON",
      "Confirm Payment\n" .
        "Policy: {$policyNo}\n" .
        "Amount: GHS {$premium}\n" .
        "Network: {$net}\n\n" .
        "1. Confirm\n" .
        "2. Cancel\n" .
        "0. Back to Menu"
    );
  }

  if (($draft["step"] ?? "") === "PAY_CONFIRM") {
    $c = $lastInput;

    if ($c === "0") go_home($con, $sessionId);

    if ($c === "2") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Cancelled.");
    }

    if ($c !== "1") {
      ussd_response("CON", "1. Confirm\n2. Cancel\n0. Back to Menu");
    }

    $draft = reload_draft($con, $sessionId);

    $policyNo = (string)($draft["pay_policy_number"] ?? "");
    $amount   = (string)($draft["total_premium"] ?? "");
    $net      = (string)($draft["momo_network"] ?? "");

    if ($policyNo === "" || $amount === "" || $net === "") {
      draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
      ussd_response("END", "Missing payment details. Please try again.");
    }

    $payResp = post_json($PAY_PREMIUM_API, [
      "policy_number" => $policyNo,
      "mobil_no"      => (string)$phoneNumber,
      "amount"        => $amount,
      "momo_network"  => $net
    ]);

    draft_update($con, $sessionId, ["step" => "START", "flow" => "", "pay_policies_json" => null]);

    if (!($payResp["ok"] ?? false)) {
      $dump = json_encode($payResp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
      ussd_response("END", "Payment failed.\n{$dump}");
    }

    $refNo = (string)($payResp["refNo"] ?? "");
    $msg   = (string)($payResp["gateway_response"]["responseMessage"] ?? "Processing payment");

    ussd_response("END", "Payment initiated.\n{$msg}\nRef: {$refNo}");
  }

  ussd_response("END", "Session error. Please try again.");
}

/**
 * ==========================
 * ✅ REGISTER FLOW (1)
 * ==========================
 */
if ($menuChoice !== "1") {
  ussd_response("END", "Invalid option.");
}

/** START -> ask full name */
if ($step === "START") {
  draft_update($con, $sessionId, ["step" => "ASK_FULL_NAME", "flow" => "REGISTER"]);
  ussd_response("CON", "Register\nEnter full name:");
}

/** Full name */
if ($step === "ASK_FULL_NAME") {
  $fullName = $inputs[count($inputs) - 1] ?? "";
  if ($fullName === "" || $fullName === "1") ussd_response("CON", "Register\nEnter full name:");

  draft_update($con, $sessionId, ["full_name" => $fullName, "step" => "ASK_DOB", "flow" => "REGISTER"]);
  ussd_response("CON", "Enter date of birth (YYYY-MM-DD):");
}

/** DOB -> call AGE API */
if ($step === "ASK_DOB") {
  $dob = $inputs[count($inputs) - 1] ?? "";
  if ($dob === "" || $dob === "1") ussd_response("CON", "Enter date of birth (YYYY-MM-DD):");

  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $dob)) {
    ussd_response("CON", "Invalid format. Enter DOB as YYYY-MM-DD:");
  }

  $ageResp = post_json($AGE_API, [
    "life_type" => "Main Life",
    "date_of_birth" => $dob
  ]);

  if (!($ageResp["ok"] ?? false)) {
    ussd_response("CON", "Invalid date of birth. Enter again (YYYY-MM-DD):");
  }

  draft_update($con, $sessionId, ["date_of_birth" => $dob, "step" => "ASK_GENDER", "flow" => "REGISTER"]);
  ussd_response("CON", "Select gender:\n1. Male\n2. Female");
}

/** Gender */
if ($step === "ASK_GENDER") {
  $g = $inputs[count($inputs) - 1] ?? "";
  if ($g === "1") $gender = "Male";
  elseif ($g === "2") $gender = "Female";
  else ussd_response("CON", "Select gender:\n1. Male\n2. Female");

  draft_update($con, $sessionId, ["gender" => $gender, "step" => "ASK_ADD_DEP", "flow" => "REGISTER"]);
  ussd_response("CON", "Do you want to add dependence?\n1. Yes\n2. No");
}

/** Add dependent? */
if ($step === "ASK_ADD_DEP") {
  $yn = $inputs[count($inputs) - 1] ?? "";

  if ($yn === "1") {
    draft_update($con, $sessionId, ["add_dep" => "Y", "step" => "ASK_DEP_COUNT", "flow" => "REGISTER"]);
    ussd_response("CON", "Enter number of dependence (1 or 2):");
  }

  if ($yn === "2") {
    draft_update($con, $sessionId, [
      "add_dep" => "N",
      "dep_count" => "0",
      "dep_index" => "1",
      "dep_full_name" => json_encode([]),
      "dep_date_of_birth" => json_encode([]),
      "dep_gender" => json_encode([]),
      "dep_relationship" => json_encode([]),
      "flow" => "REGISTER"
    ]);

    show_products($con, $sessionId, $PRODUCT_API);
  }

  ussd_response("CON", "Do you want to add dependence?\n1. Yes\n2. No");
}

/** Dep count */
if ($step === "ASK_DEP_COUNT") {
  $count = (int)($inputs[count($inputs) - 1] ?? 0);
  if (!in_array($count, [1, 2], true)) {
    ussd_response("CON", "Enter number of dependence (1 or 2):");
  }

  draft_update($con, $sessionId, [
    "dep_count" => (string)$count,
    "dep_index" => "1",
    "dep_full_name" => json_encode([]),
    "dep_date_of_birth" => json_encode([]),
    "dep_gender" => json_encode([]),
    "dep_relationship" => json_encode([]),
    "step" => "ASK_DEP_NAME",
    "flow" => "REGISTER"
  ]);

  ussd_response("CON", "Dependent 1\nEnter full name:");
}

/** Dep name */
if ($step === "ASK_DEP_NAME") {
  $draft = reload_draft($con, $sessionId);
  $depIndex = (int)($draft["dep_index"] ?? 1);

  $name = $inputs[count($inputs) - 1] ?? "";
  if ($name === "" || $name === "1") {
    ussd_response("CON", "Dependent {$depIndex}\nEnter full name:");
  }

  $names = json_decode($draft["dep_full_name"] ?? "[]", true) ?: [];
  $names[$depIndex - 1] = $name;

  draft_update($con, $sessionId, [
    "dep_full_name" => json_encode($names),
    "step" => "ASK_DEP_DOB",
    "flow" => "REGISTER"
  ]);

  ussd_response("CON", "Dependent {$depIndex}\nEnter date of birth (YYYY-MM-DD):");
}

/** Dep DOB */
if ($step === "ASK_DEP_DOB") {
  $draft = reload_draft($con, $sessionId);
  $depIndex = (int)($draft["dep_index"] ?? 1);

  $dob = $inputs[count($inputs) - 1] ?? "";
  if ($dob === "") ussd_response("CON", "Dependent {$depIndex}\nEnter date of birth (YYYY-MM-DD):");

  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $dob)) {
    ussd_response("CON", "Invalid format.\nDependent {$depIndex}\nEnter DOB as YYYY-MM-DD:");
  }

  $ageResp = post_json($AGE_API, [
    "life_type" => "Secondary Life",
    "date_of_birth" => $dob
  ]);

  if (!($ageResp["ok"] ?? false)) {
    ussd_response("CON", "Invalid DOB.\nDependent {$depIndex}\nEnter date of birth (YYYY-MM-DD):");
  }

  $dobs = json_decode($draft["dep_date_of_birth"] ?? "[]", true) ?: [];
  $dobs[$depIndex - 1] = $dob;

  draft_update($con, $sessionId, [
    "dep_date_of_birth" => json_encode($dobs),
    "step" => "ASK_DEP_GENDER",
    "flow" => "REGISTER"
  ]);

  ussd_response("CON", "Dependent {$depIndex}\nSelect gender:\n1. Male\n2. Female");
}

/** Dep gender */
if ($step === "ASK_DEP_GENDER") {
  $draft = reload_draft($con, $sessionId);
  $depIndex = (int)($draft["dep_index"] ?? 1);

  $g = $inputs[count($inputs) - 1] ?? "";
  if ($g === "1") $gender = "Male";
  elseif ($g === "2") $gender = "Female";
  else ussd_response("CON", "Dependent {$depIndex}\nSelect gender:\n1. Male\n2. Female");

  $genders = json_decode($draft["dep_gender"] ?? "[]", true) ?: [];
  $genders[$depIndex - 1] = $gender;

  draft_update($con, $sessionId, [
    "dep_gender" => json_encode($genders),
    "step" => "ASK_DEP_REL",
    "flow" => "REGISTER"
  ]);

  ussd_response(
    "CON",
    "Dependent {$depIndex}\n" .
      "Select relationship:\n" .
      "1. Mother\n" .
      "2. Father\n" .
      "3. Father-in-law\n" .
      "4. Mother-in-law\n"
  );
}

/** Dep relationship */
if ($step === "ASK_DEP_REL") {
  $draft = reload_draft($con, $sessionId);
  $depIndex = (int)($draft["dep_index"] ?? 1);
  $depCount = (int)($draft["dep_count"] ?? 0);

  $rel = $inputs[count($inputs) - 1] ?? "";

  $relationshipMap = [
    "1" => "Mother",
    "2" => "Father",
    "3" => "Father-in-law",
    "4" => "Mother-in-law"
  ];

  if (!isset($relationshipMap[$rel])) {
    ussd_response(
      "CON",
      "Dependent {$depIndex}\n" .
        "Invalid selection. Choose 1-4:\n" .
        "1. Mother\n2. Father\n3. Father-in-law\n4. Mother-in-law\n"
    );
  }

  $relationship = $relationshipMap[$rel];

  $rels = json_decode($draft["dep_relationship"] ?? "[]", true) ?: [];
  $rels[$depIndex - 1] = $relationship;

  draft_update($con, $sessionId, [
    "dep_relationship" => json_encode($rels),
    "flow" => "REGISTER"
  ]);

  if ($depIndex < $depCount) {
    $next = $depIndex + 1;
    draft_update($con, $sessionId, [
      "dep_index" => (string)$next,
      "step" => "ASK_DEP_NAME",
      "flow" => "REGISTER"
    ]);
    ussd_response("CON", "Dependent {$next}\nEnter full name:");
  }

  show_products($con, $sessionId, $PRODUCT_API);
}

/** Select product */
if ($step === "SELECT_PRODUCT") {
  $choice = (int)($inputs[count($inputs) - 1] ?? 0);
  if ($choice <= 0) ussd_response("CON", "Select Product:\nChoose a number.");

  $prodResp = get_json($PRODUCT_API);
  if (!($prodResp["ok"] ?? false)) ussd_response("END", "Unable to load products.");

  $products = $prodResp["products"] ?? [];
  if ($choice > count($products)) ussd_response("CON", "Invalid selection. Try again.");

  $selected = $products[$choice - 1];
  $productIndex = (int)$selected["product_index"];
  $productName  = (string)$selected["product_name"];

  draft_update($con, $sessionId, [
    "product_index" => (string)$productIndex,
    "product_name" => $productName,
    "step" => "SELECT_SUM_ASSURED",
    "flow" => "REGISTER"
  ]);

  $saResp = post_json($SUM_ASSURED_API, ["product_index" => $productIndex]);
  if (!($saResp["ok"] ?? false)) ussd_response("END", "Unable to load sum assured.");

  $sas = $saResp["sum_assured"] ?? [];
  if (count($sas) === 0) ussd_response("END", "No sum assured available.");

  $menu = "Select Sum Assured:\n";
  foreach ($sas as $i => $val) {
    $n = $i + 1;
    $menu .= "{$n}. {$val}\n";
  }

  ussd_response("CON", trim($menu));
}

/** Select sum assured */
if ($step === "SELECT_SUM_ASSURED") {
  $draft = reload_draft($con, $sessionId);

  $choice = (int)($inputs[count($inputs) - 1] ?? 0);
  if ($choice <= 0) ussd_response("CON", "Select Sum Assured:\nChoose a number.");

  $productIndex = (int)($draft["product_index"] ?? 0);
  if ($productIndex <= 0) ussd_response("END", "Product missing.");

  $saResp = post_json($SUM_ASSURED_API, ["product_index" => $productIndex]);
  if (!($saResp["ok"] ?? false)) ussd_response("END", "Unable to load sum assured.");

  $sas = $saResp["sum_assured"] ?? [];
  if ($choice > count($sas)) ussd_response("CON", "Invalid selection. Try again.");

  $sumAssured = (int)$sas[$choice - 1];

  $depCount = (int)($draft["dep_count"] ?? 0);
  $noParam  = ($depCount > 0) ? (string)$depCount : "";

  $premResp = post_json($PREMIUM_API, [
    "product_index" => (string)$productIndex,
    "sum_assured"   => (string)$sumAssured,
    "no"            => $noParam
  ]);

  if (!($premResp["ok"] ?? false)) {
    $dump = json_encode($premResp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ussd_response("END", "Failed: Premium API returned ok=false\n{$dump}");
  }

  $premium = (float)($premResp["total_premium"] ?? 0);

  draft_update($con, $sessionId, [
    "sum_assured" => (string)$sumAssured,
    "total_premium" => (string)$premium,
    "step" => "ASK_MOMO_NETWORK",
    "flow" => "REGISTER"
  ]);

  ussd_response(
    "CON",
    "Premium: GHS " . number_format($premium, 2) . "\n" .
      "Select Mobile Network:\n1. MTN\n2. Telecel\n3. AirtelTigo"
  );
}

/** Ask momo network */
if ($step === "ASK_MOMO_NETWORK") {
  $ch = $inputs[count($inputs) - 1] ?? "";
  if ($ch === "1") $net = "MTN";
  elseif ($ch === "2") $net = "Telecel";
  elseif ($ch === "3") $net = "AirtelTigo";
  else ussd_response("CON", "Select Mobile Network:\n1. MTN\n2. Telecel\n3. AirtelTigo");

  draft_update($con, $sessionId, [
    "momo_network" => $net,
    "mobile_money_number" => $phoneNumber,
    "step" => "CONFIRM_SUBMIT",
    "flow" => "REGISTER"
  ]);

  $draft = reload_draft($con, $sessionId);

  ussd_response(
    "CON",
    "Confirm Registration\n" .
      "Name: " . ($draft["full_name"] ?? "") . "\n" .
      "Product: " . ($draft["product_name"] ?? "") . "\n" .
      "Sum Assured: " . ($draft["sum_assured"] ?? "") . "\n" .
      "Premium: GHS " . ($draft["total_premium"] ?? "") . "\n\n" .
      "1. Confirm\n2. Cancel"
  );
}

/** Confirm submit -> call signup API */
if ($step === "CONFIRM_SUBMIT") {
  $draft = reload_draft($con, $sessionId);
  $c = $inputs[count($inputs) - 1] ?? "";

  if ($c === "2") {
    draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);
    ussd_response("END", "Cancelled.");
  }

  if ($c !== "1") {
    ussd_response("CON", "1. Confirm\n2. Cancel");
  }

  $depCount = (int)($draft["dep_count"] ?? 0);

  $depNames = json_decode($draft["dep_full_name"] ?? "[]", true) ?: [];
  $depDobs  = json_decode($draft["dep_date_of_birth"] ?? "[]", true) ?: [];
  $depGens  = json_decode($draft["dep_gender"] ?? "[]", true) ?: [];
  $depRels  = json_decode($draft["dep_relationship"] ?? "[]", true) ?: [];

  $depMomos = ($depCount > 0) ? array_fill(0, $depCount, (string)$phoneNumber) : [];

  $payload = [
    "mobile_money_number" => (string)($draft["mobile_money_number"] ?? $phoneNumber),
    "full_name" => (string)($draft["full_name"] ?? ""),
    "date_of_birth" => (string)($draft["date_of_birth"] ?? ""),
    "gender" => (string)($draft["gender"] ?? ""),
    "sum_assured" => (int)($draft["sum_assured"] ?? 0),
    "product_id" => (int)($draft["product_index"] ?? 0),
    "product_name" => (string)($draft["product_name"] ?? ""),
    "total_premium" => (float)($draft["total_premium"] ?? 0),
    "momo_network" => (string)($draft["momo_network"] ?? ""),

    "addlife_mobile_money_number" => $depMomos,
    "addlife_full_name" => $depNames,
    "addlife_date_of_birth" => $depDobs,
    "addlife_gender" => $depGens,
    "addlife_relationship" => $depRels
  ];

  $resp = post_json($SIGNUP_API, $payload);

  if (!($resp["ok"] ?? false)) {
    $dump = json_encode($resp, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    ussd_response("END", "Signup failed.\n{$dump}");
  }

  $subId = $resp["subscriptions_id"] ?? "";
  $prem  = $draft["total_premium"] ?? "";
  $addInserted = $resp["additional_lives_inserted"] ?? "";

  draft_update($con, $sessionId, ["step" => "START", "flow" => ""]);

  $extra = ($addInserted !== "") ? "\nAdditional Lives: {$addInserted}" : "";
  ussd_response("END", "Success! You will receive a prompt shortly.\nSubscription ID: {$subId}\nPremium: GHS {$prem}{$extra}");
}

// Fallback
ussd_response("END", "Session error. Please try again.");
?>